/*
 * ffproxy (c) 2002, 2003 Niklas Olmes <niklas@noxa.de>
 *                                     <niklas.olmes@web.de>
 * http://faith.eu.org
 * 
 * $Id: cache.c,v 1.6 2003/07/20 10:38:23 niklas Exp $
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 675
 * Mass Ave, Cambridge, MA 02139, USA.
 */

#include <sys/types.h>
#include <sys/stat.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef LINUX
#define __USE_GNU 1
#include <asm/fcntl.h>
#else
#include <fcntl.h>
#endif
#include <unistd.h>
#include <ctype.h>

#include "cfg.h"
#include "req.h"
#include "print.h"
#include "time.h"
#include "cache.h"

#define MODE_FILE 0640

static int      mktree(char *);
static int      url_to_file(const char *, char *, size_t);
static int      openfile(const struct req *, int);
static int      lock(const struct req *);
static int      islocked(const struct req *);

static const char simple_header[] = "HTTP/1.0 200 OK (cached)\r\nProxy-Connection: close\r\nConnection: close\r\nContent-Type: ";
static const char default_mime[] = "application/octet-stream";
static const char cache_dir[] = "cache/";
static const char http_index[] = "index.html";

static const struct mt {
	const char     *ext, *ctype;
}               mimetypes[] = {
	{
		"html", "text/html"
	},
	{
		"htm", "text/html"
	},
	{
		"css", "text/css"
	},
	{
		"gif", "image/gif"
	},
	{
		"png", "image/png"
	},
	{
		"jpeg", "image/jpeg"
	},
	{
		"jpg", "image/jpeg"
	},
	{
		"avi", "video/x-msvideo"
	},
	{
		"mpeg", "video/mpeg"
	},
	{
		"mp3", "audio/mpeg"
	},
	{
		"ogg", "audio/x-oggvorbis"
	},
	{
		"dvi", "application/x-dvi"
	},
	{
		"ps", "application/postscript"
	},
	{
		"pdf", "application/pdf"
	},
	{
		NULL, NULL
	}
};

extern struct cfg config;

int
deliverfromcache(int cl, int f, const struct req * r)
{
	char            buf[4096];
	int             len;

	if (!config.cache)
		return -1;

	if (!config.online) {
		char            line[256];
		int             p, i;
		struct stat     fstat;

		(void) strcpy(buf, simple_header);

		p = strlen(r->fname);
		while (p > 0 && r->fname[p] != '.' && r->fname[p] != '/')
			p--;
		i = -1;
		if (r->fname[p] == '.') {
			p++;
			i = 0;
			while (mimetypes[i].ext != NULL)
				if (strcmp(mimetypes[i++].ext, r->fname + p) == 0)
					break;
		}
		if (i >= 0 && mimetypes[i].ext != NULL)
			(void) strcat(buf, mimetypes[i].ctype);
		else
			(void) strcat(buf, default_mime);

		(void) strcat(buf, "\r\n");

		if (stat(r->fname, &fstat) == 0) {
			(void) sprintf(line,
				       "Server: ffproxy\r\n"
				       "Connection: close\r\n"
				       "Accept-Ranges: bytes\r\n"
				       "Content-Length: %ld\r\n"
				       "Last-Modified: ",
				       (long) fstat.st_size);
			(void) strcat(buf, line);
			(void) strftime(line, sizeof(line),
					"%a, %d %b %Y %T GMT",
					gmtime(&fstat.st_mtime));
			(void) strcat(buf, line);
			(void) strcat(buf, "\r\n");
		}
		(void) strcat(buf, "\r\n");

		debug("deliverfromcache() => our header is (%s)", buf);
		(void) write(cl, buf, strlen(buf));
	}
	while ((len = read(f, buf, sizeof(buf))) > 0)
		if (write(cl, buf, len) < 0) {
			(void) close(f);
			(void) close(cl);
			return -1;
		}
	(void) close(f);

	return 0;
}

int
havecached(const struct req * r)
{
	struct stat     fstat;
	time_t          remote, local;

	if (!config.cache)
		return 0;

	if (stat(r->fname, &fstat) != 0)
		return 0;

	local = fstat.st_mtime;
	remote = my_convtime(r->tstamp);

	debug("havecached() => times local (%d) remote (%d)",
	      local, remote);
	debug("havecached() => sizes local (%ld) remote (%ld)",
	      (long) fstat.st_size, r->clen);

	if (local != remote) {
		debug("havecached() => times differ");
		return 0;
	}
	if (fstat.st_size != r->clen) {
		debug("havecached() => sizes differ");
		return 0;
	}
	return 1;
}

int
opencache(struct req * r, int create)
{
	char            buf[512];

	if (!config.cache)
		return -1;

	if (url_to_file(r->url, buf, sizeof(buf)) != 0)
		return -1;

	if (config.online && create)
		if (mktree(buf) != 0)
			return -1;

	(void) strcpy(r->fname, buf);
	(void) strcpy(r->lname, buf);
	r->lname[strlen(buf)] = ',';
	r->lname[strlen(buf) + 1] = '\0';

	debug("opencache() => created lock file name (%s) for (%s)", r->lname, r->fname);

	return openfile(r, create);
}

static int
lock(const struct req * r)
{
	if (link(r->fname, r->lname) != 0) {
		debug("lock() => failed to lock file (%s)", r->fname);
		return 1;
	}
	debug("lock() => LOCKED file (%s)", r->fname);

	return 0;
}

int
unlock(const struct req * r)
{
	if (unlink(r->lname) != 0) {
		debug("unlock() => failed to remove lock for file (%s)", r->fname);
		return 1;
	}
	debug("lock() => UN-LOCKED file (%s)", r->fname);

	return 0;
}

static int
islocked(const struct req * r)
{
	struct stat     s;

	if (stat(r->lname, &s) == 0) {
		debug("islocked() => file (%s) is locked", r->fname);
		return 1;
	}
	return 0;
}

void
closecache(int f, const struct req * r)
{
	(void) unlock(r);
	(void) close(f);
}

static int
mktree(char *path)
{
	size_t          i, len;

	len = strlen(path);
	i = 0;
	if (path[i++] == '/' || len < 2)
		return -1;

	while (i < len) {
		if (path[i] == '/') {
			path[i] = '\0';
			debug("mktree() => mkdir (%s)", path);
			(void) mkdir(path, 0750);
			path[i] = '/';
		}
		i++;
	}

	return 0;
}

static int
url_to_file(const char *url, char *buf, size_t size)
{
	size_t          i, j;
	int             passed_host;
	char            last;

	last = '.';
	j = strlen("http://");
	if (strlen(url) <= j || j >= size || url[j] == '/')
		return -1;
	if (strlen(cache_dir) >= size)
		return -1;
	(void) strcpy(buf, cache_dir);
	i = strlen(cache_dir);
	passed_host = 0;
	while (i < size - 1 && url[j] != '\0' &&
	       (isalnum(url[j]) || url[j] == '.' || url[j] == ','
		|| url[j] == ':' || url[j] == '-' || url[j] == '_'
		|| url[j] == '/')) {

		if (last == '.' && url[j] == '.') {
			buf[i] = '\0';
			return -1;
		}
		last = url[j];

		if (!passed_host && url[j] == '/')
			passed_host = 1;

		buf[i++] = url[j++];
	}
	buf[i] = '\0';

	if ((i < 2 || url[j] != '\0') && url[j] != '#')
		return -1;

	if (buf[i - 1] == '/' || !passed_host) {
		if (i + 2 + strlen(http_index) >= size)
			return -1;
		if (!passed_host) {
			buf[i++] = '/';
			buf[i] = '\0';
		}
		strcat(buf, http_index);
	}
	debug("url_to_file() => filename (%s)", buf);

	return 0;
}

static int
openfile(const struct req * r, int create)
{
	int             f;

	debug("openfile() => opening (%s) mode %d", r->fname, create);

	if (islocked(r)) {
		debug("openfile() => file (%s) is locked", r->fname);
		return -1;
	}
	if ((f = open(r->fname, create ?
		      (O_CREAT | O_WRONLY | O_TRUNC | O_NOFOLLOW)
		      : (O_RDONLY | O_NOFOLLOW), MODE_FILE)) < 0) {
		debug("openfile() => error opening file (%s)", r->fname);
		return -1;
	}
	if (create)
		(void) lock(r);

	return f;
}
