/*
 * ffproxy (c) 2002, 2003 Niklas Olmes <niklas@noxa.de>
 *                                     <niklas.olmes@web.de>
 * http://faith.eu.org
 * 
 * $Id: http.c,v 1.4 2003/05/12 00:41:40 niklas Exp $
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 675
 * Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>
#include <ctype.h>

#include "req.h"
#include "print.h"
#include "http.h"
#include "cfg.h"

static const char http_get[] = "GET ";
static const char http_post[] = "POST ";
static const char http_head[] = "HEAD ";
static const char http[] = "http://";
static const char httpv[] = "HTTP/";

int
http_url(struct req * r, const char *s)
{
	size_t          i, k;
	char           *p;
	extern struct cfg config;
	char		accelport[10];

	if (strncmp(http_get, s, strlen(http_get)) == 0) {
		r->type = GET;
		s += strlen(http_get);
	} else if (strncmp(http_post, s, strlen(http_post)) == 0) {
		r->type = POST;
		s += strlen(http_post);
	} else if (strncmp(http_head, s, strlen(http_head)) == 0) {
		r->type = HEAD;
		s += strlen(http_head);
	} else {
		r->type = UNKNOWN;
		return -1;
	}

	while (*s == ' ')
		s++;

	debug("http_url() => got url part (%s)", s);

	if (config.use_accel) {
		debug("http_url() => using as accelerator proxy");
		i = 0;
debug("http: %s (%d)", http, strlen(http));
debug("config.accelhost: %s (%d)", config.accelhost, strlen(config.accelhost));
		// add http://
		for (k=0; k<=strlen(http)-1; k++) {
			r->url[i++] = http[k];
		}
		// add hostname (or IP)
		for (k=0; k<=strlen(config.accelhost)-1; k++) {
			r->url[i++] = config.accelhost[k];
		}
		// add port number
		r->url[i++] = ':';
		snprintf(accelport, sizeof(accelport), "%d", config.accelport);
		for (k=0; k<=strlen(accelport)-1; k++) {
			r->url[i++] = accelport[k];
		}
		r->url[i] = '\0';
		debug("http_url() => accelerator mode created url (%s)", r->url);
	} else {
		if (strncmp(s, http, strlen(http)) != 0) {
			r->type = UNKNOWN;
			return -1;
		}

		i = 0;
		while (i < strlen(http)) {
			r->url[i] = http[i];
			i++, s++;
		}

		while (i < sizeof(r->url) - 1 && *s != '_'
		       && (isalnum(*s) || *s == '-' || *s == '.' || *s == ':'))
			r->url[i++] = tolower(*(s++));
		r->url[i] = '\0';
		if (*s != '/' && *s != ' ') {
			r->type = UNKNOWN;
			return -1;
		}
	}
		
	k = 0;
	while (i < sizeof(r->url) - 1 && k < sizeof(r->urlpath) - 1 && *s != ' ' && *s != '\0' && isprint(*s)) {
		r->urlpath[k++] = *s;
		r->url[i++] = *(s++);
	}
	if (k == 0)
		r->urlpath[k++] = '/';
	r->urlpath[k] = '\0';
	r->url[i] = '\0';
	if (*s != ' ') {
		r->type = UNKNOWN;
		return -1;
	}

	debug("http_url() => extracted urlpath (%s)", r->urlpath);
	debug("http_url() => extracted url (%s)", r->url);

	while (*s == ' ')
		s++;

	debug("http_url() => got version part (%s)", s);

	if (strncasecmp(s, httpv, strlen(httpv)) != 0)
		return -1;
	s += strlen(httpv);
	r->vmajor = 0;
	r->vminor = 0;
	i = 0;
	while (i < 2 && *s != '\0' && isdigit(*s)) {
		r->vmajor = r->vmajor * 10 + (*(s++) - '0');
		i++;
	}
	if (*s == '.') {
		s++;
		while (i < 2 && *s != '\0' && isdigit(*s)) {
			r->vminor = r->vminor * 10 + (*(s++) - '0');
			i++;
		}
	}
	debug("http_url() => got type %d url (%s) version maj %d min %d",
	      r->type, r->url, r->vmajor, r->vminor);

	p = r->url;
	p += strlen(http);

	i = 0;
	while ((isalnum(*p) || *p == '-' || *p == '.') && i < sizeof(r->host) - 1)
		r->host[i++] = tolower(*(p++));
	r->host[i] = '\0';

	if (i >= sizeof(r->host) - 1) {
		debug("http_url() => host: too long (%s)", r->host);
		*r->host = '\0';
		r->port = 0;
		return -1;
	}
	if (*p == ':') {
		p++;
		r->port = 0;
		while (isdigit(*p)) {
			r->port = r->port * 10 + (*(p++) - '0');
			if (r->port >= 65534) {
				debug("http_url() => port: bad port number");
				r->port = 0;
				return -1;
			}
		}
		if (*p != '\0' && *p != ' ' && *p != '/') {
			debug("http_url() => port: bad port");
			r->port = 0;
			return -1;
		}
		debug("http_url() => port: %d", r->port);
	} else {
		debug("http_url() => default port 80");
		r->port = 80;
	}

	return 0;
}

static const char http_clen[] = "Content-Length: ";
static const char http_tstamp[] = "Last-Modified: ";

int
http_parse(struct req * r, const char *s)
{
	size_t          i;

	if (strncasecmp(http_clen, s, strlen(http_clen)) == 0) {
		debug("http_parse() => found clen header (%s)", s);

		s += strlen(http_clen);

		while (isblank(*s))
			s++;

		if (!isdigit(*s)) {
			debug("http_parse() => clen: no digit found (%s)", s);
			return -1;
		}
		r->clen = 0L;
		i = 0;
		while (i < 10 && isdigit(*s))
			r->clen = r->clen * 10L + (long) (*(s++) - '0');

		if (*s != '\0') {
			r->clen = 0L;
			debug("http_parse() => clen: too long");
			return -1;
		}
		debug("http_parse() => clen: %ld bytes", r->clen);
		return 0;
	} else if (strncasecmp(http_tstamp, s, strlen(http_tstamp)) == 0) {
		debug("http_parse() => found tstamp header (%s)", s);

		s += strlen(http_tstamp);

		while (isblank(*s))
			s++;

		i = 0;
		while (i < sizeof(r->tstamp) - 1 && *s != '\0')
			r->tstamp[i++] = *(s++);
		r->tstamp[i] = '\0';

		if (*s != '\0') {
			r->tstamp[0] = '\0';
			debug("http_parse()_ => tstamp: too long");
			return -1;
		}
		debug("http_parse() => tstamp: extracted (%s)", r->tstamp);
		return 0;
	}
	return 1;
}
