/*
 * ffproxy (c) 2002, 2003 Niklas Olmes <niklas@noxa.de>
 *                                     <niklas.olmes@web.de>
 * http://faith.eu.org
 * 
 * $Id: main.c,v 1.41 2003/08/01 20:11:05 niklas Exp niklas $
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 675
 * Mass Ave, Cambridge, MA 02139, USA.
 */

#include <sys/types.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
#include <pwd.h>

#include "cfg.h"
#include "print.h"
#include "socket.h"
#include "db.h"
#include "dns.h"
#include "signals.h"

static void     usage(void);
static void     drop_privileges(void);

static const char version[] = "1.4.1+accel";
static const char rcsid[] = "$Id: main.c,v 1.41 2003/08/01 20:11:05 niklas Exp niklas $";
char            loop_header[100];

struct cfg      config;

int
main(int argc, char *argv[])
{
	int             c;

	(void) memset(&config, 0, sizeof(config));
	config.ip = 0L;
	config.port = 0;
	config.daemon = 0;
	config.childs = 10;
	config.ccount = 0;
	config.backlog = 0;
	config.uid = 0;
	config.gid = 0;
	*config.chroot = '\0';
	(void) strncpy(config.dbdir, DATADIR, sizeof(config.dbdir) - 1);
	config.dbdir[sizeof(config.dbdir) - 1] = '\0';
	(void) strncpy(config.file, CFGFILE, sizeof(config.file) - 1);
	config.file[sizeof(config.file) - 1] = '\0';
	*config.proxyhost = '\0';
	config.proxyport = 0;
	config.syslog = 1;
	config.logrequests = 0;
	config.online = 1;
	config.cache = 0;
	config.cache_max_file_size = 1024 * 2;
	config.use_ipv6 = 1;
	config.aux_proxy_ipv6 = 1;
	config.use_accel = 0;
	config.accelport = 80;

	while ((c = getopt(argc, argv, "vdc:p:x:X:l:u:g:r:D:f:s4ha:A:")) != -1) {
		switch (c) {
		case 'v':
			(void) printf("ffproxy version %s, %s\n",
				      version, rcsid);
			exit(0);
			break;
		case 'd':
			config.daemon = 1;
			break;
		case 'c':
			config.ip = resolve(optarg);
			break;
		case 'p':
			config.port = atoi(optarg);
			break;
		case 'x':
			(void) strncpy(config.proxyhost, optarg, sizeof(config.proxyhost) - 1);
			config.proxyhost[sizeof(config.proxyhost) - 1] = '\0';
			break;
		case 'X':
			config.proxyport = atoi(optarg);
			break;
		case 'l':
			config.childs = atoi(optarg);
			break;
		case 'u':
			config.uid = atoi(optarg);
			break;
		case 'g':
			config.gid = atoi(optarg);
			break;
		case 'r':
			(void) strncpy(config.chroot, optarg, sizeof(config.chroot) - 1);
			config.chroot[sizeof(config.chroot) - 1] = '\0';
			break;
		case 'D':
			(void) strncpy(config.dbdir, optarg, sizeof(config.dbdir) - 1);
			config.dbdir[sizeof(config.dbdir) - 1] = '\0';
			break;
		case 'f':
			(void) strncpy(config.file, optarg, sizeof(config.file) - 1);
			config.file[sizeof(config.file) - 1] = '\0';
			break;
		case 's':
			config.syslog = 0;
			break;
		case '4':
			config.use_ipv6 = 0;
			break;
		case 'a':
			(void) strncpy(config.accelhost, optarg, sizeof(config.accelhost) - 1);
			config.accelhost[sizeof(config.accelhost) - 1] = '\0';
			config.use_accel = 1;
			break;
		case 'A':
			config.accelport = atoi(optarg);
			break;
		case 'h':
		default:
			usage();
			/* NOTREACHED */
			break;
		}
	}
	argc -= optind;
	argv += optind;

	setup_log_master();
	info("started, initializing");
	load_databases();
	drop_privileges();

	if (config.daemon) {
		FILE           *fp;

		if (daemon(1, 0) != 0)
			fatal("daemon() failed");
		(void) close(0);
		(void) close(1);
		(void) close(2);

		(void) chdir(config.dbdir);
		if ((fp = fopen("ffproxy.pid", "w")) == NULL)
			fatal("cannot create pid file ffproxy.pid in %s", config.dbdir);

		(void) fprintf(fp, "%ld", (long) getpid());
		(void) fclose(fp);
	}
	(void) snprintf(loop_header, sizeof(loop_header), "X-Loop-%d-%d: true", getpid(), time(NULL));

	init_sighandlers();
	open_socket();

	/* NOTREACHED */
	return 0;
}

static void
usage(void)
{
	(void) fprintf(stderr, "ffproxy %s -- (c) 2002, 2003 Niklas Olmes <niklas@noxa.de>\n", version);
	(void) fprintf(stderr, "                                           <niklas.olmes@web.de>\n");
	(void) fprintf(stderr, "   GNU GPL.  Website: http://faith.eu.org/programs.html\n");
	(void) fprintf(stderr,
		       "usage: ffproxy [-vds4h] [-c host|ip] [-p port] [-x host|ip] [-X port] [-l max]\n"
	      "               [-u uid -g gid] [-r dir] [-D dir] [-f file]\n"
	      "               [-a host|ip] [-A port]\n"
		       "\n"
		       " -v      print version number\n"
		       " -d      become daemon\n"
		       " -s      silent.  don't log to syslog.\n"
		       " -4      Use IPv4 only.  Don't try contacting via IPv6.\n"
		       " -h      usage (this screen)\n"
		       "\n"
		       " -c host|ip  bind to IP\n"
		       " -p port     bind to port\n"
		       " -x host|ip  auxiliary forward proxy\n"
		       " -X port     auxiliary forward proxy port\n"
		       " -l max      maximum number of concurrent requests\n"
		       " -u uid      change uid\n"
		       " -g gid      change gid\n"
		       " -r dir      chroot to dir\n"
		       " -D dir      databases are in dir (default is %s)\n"
		       " -f file     use config file (default is %s)\n"
		       " -a host|ip  auxiliary forward server to use\n"
		       " -A port     auxiliary forward server port\n"
			DATADIR, CFGFILE);
	exit(1);
}

static void
drop_privileges(void)
{
	extern struct cfg config;
	struct passwd  *pwd;

	if (config.uid == 0 || config.gid == 0) {
		info("not changing UID/GID");
	} else {
		pwd = getpwuid(config.uid);

		if (*config.chroot != '\0') {
			if (chdir(config.chroot) != 0)
				fatal("chdir() failed");
			if (chroot(config.chroot) != 0)
				fatal("chroot() failed");
			info("chroot()ed to %s\n", config.chroot);
		}
		if (setgid(config.gid) != 0)
			fatal("setgid() failed");
		if (initgroups(pwd->pw_name, config.gid) != 0)
			fatal("initgroups() failed");

		if (setuid(config.uid) != 0)
			fatal("setuid() failed");

	}

	info("=> UID(%d), EUID(%d), GID(%d), EGID(%d)", getuid(), geteuid(), getgid(), getegid());
}
